<?php

/* Maddos Theme custom feature functions */

/**
 * check for CCBill type linking '...html=http://www.domain.com/main.htm'
 */
function maddos_check_ccbill_link( $link ) {
  $pos = stripos( $link, 'html=' );
  if ( false !== $pos ) {
	$link = substr( $link, $pos+5 );
	$amp = strpos( $link, '&' );
	if ( $amp ) $link = substr( $link, 0, $amp );
  }
  return $link;
}

/**
 * Determine if url is from this domain
 */
if ( ! function_exists( 'maddos_site_url_check' ) ) :
function maddos_site_url_check( $link, $echo = true ) {
  $debug = apply_filters( 'maddos_debug', false );

  $link = apply_filters( 'maddos_site_url_check', $link );

  if ( $debug ) error_log(__FUNCTION__ . " DEBUG: ({$link})"); 

  if ( !$link ) return null;

  $key = 'maddos_' . esc_url( $link );
  $transient = get_transient( $key );
  if ( $transient && ! $debug )
	return $transient;

  $orig = $link;

  $link = maddos_check_ccbill_link( $link );

  // check for a redirected link ( affiliate url or a cloaked link for example ) up to 10 times
  // some websites do not recognize the default WordPress user-agent, so use a dummy agent...
  $args = array( 'redirection' => 0, 'user-agent' => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.8; rv:20.0) Gecko/20100101 Firefox/20.0' );

  for ( $i=0; $i < 10; $i++ ) {

	$response = wp_remote_get( esc_url( $link ), $args );

	if ( $debug ) error_log(__FUNCTION__ . " DEBUG: loop {$i} checking {$link}"); 

  	if ( is_wp_error( $response ) ) {
		if ( true === $echo ) {
			$msg = sprintf( __( "%s loop %d WP Error on %s: %s", 'maddos' ), __FUNCTION__, $i, $link, $response->get_error_message() );
			maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		}
		break;
	}

	$redirect = isset( $response['headers']['Location'] ) ? $response['headers']['Location'] : null;

	// If there were multiple Location headers, use the last header specified.
	if ( is_array( $redirect ) ) {
		$redirect = array_pop( $redirect );
	}

	if ( $debug && $redirect ) error_log(__FUNCTION__ . " DEBUG: Header Location detected, headers are: " . print_r($response['headers'],true));

	// look for mis-configured redirect response
	if ( $redirect && false !== strpos( $redirect, '127.0.0.1' ) )
		$redirect = null;

	// look for alternative methods
	if ( ! $redirect ) {
		$body = wp_remote_retrieve_body( $response );
		// look for a meta refresh
		// strip out comments, some have meta refresh in them (cnn.com)
		$body = preg_replace( '/<!--(.*?)-->/', '', $body );
		// strip out noscript as well
		$body = preg_replace( '/<noscript>(.*?)<\/noscript>/', '', $body );

		preg_match_all('/<[\s]*meta[\s]*http-equiv="?REFRESH"?' . '[\s]*content="?[0-9]*;[\s]*URL[\s]*=[\s]*([^>"]*)"?' . '[\s]*[\/]?[\s]*>/si', $body, $match);

		if (isset($match) && is_array($match) && count($match) === 2 && count($match[1]) === 1) {
			$redirect = $match[1][0];

		if ( $debug && $redirect ) error_log(__FUNCTION__ . " DEBUG: Preg match Refresh to {$link}, matches: " . print_r($match,true)); 
		}
	}

	if ( $redirect && $debug ) error_log(__FUNCTION__ . " DEBUG: redirect now " . $redirect );

	$parse = parse_url( $link );

	// what if relative redirect url? make it absolute
	if ( $redirect && false === strpos( $redirect, 'http' ) ) {
		$redirect = $parse['scheme'] . "://" . $parse['host'] . $redirect;
	}

	$redirect = maddos_check_ccbill_link( $redirect );
	$redirect = apply_filters( 'maddos_url_redirect', $redirect );

	if ( ! $redirect ) {
		if ( $debug ) error_log(__FUNCTION__ . " DEBUG: no redirect found break"); 
		break;
	}

	if ( false !== stripos( $redirect, 'localhost' ) ) {
		if ( $debug ) error_log(__FUNCTION__ . " DEBUG: localhost found break"); 
		break;
	}

	// this shouldn't be a relative or local url either
	if ( false === stripos( $redirect, '://' ) ) {
		// try to recover from last link
		if ( $debug ) error_log(__FUNCTION__ . " DEBUG: relative link fix attempt from previous link: " . print_r($parse,true));
		$url_fix = $parse['scheme'] . "://" . $parse['host'];
		$path = isset( $parse['path'] ) ? dirname( $parse['path'] ) : "";
		if ( $debug ) error_log(__FUNCTION__ . " DEBUG: path1: {$path}");
		$url_fix .= rtrim( $path, '/' ) . '/';
		if ( $debug ) error_log(__FUNCTION__ . " DEBUG: relative link fix1 {$redirect} to {$url_fix}"); 
		$url_fix .=  ltrim( $redirect, '/' );
		if ( $debug ) error_log(__FUNCTION__ . " DEBUG: relative link fix2 {$redirect} to {$url_fix}"); 
		$redirect = $url_fix;
	}

	if ( $link === $redirect ) { // avoid sites that loop upon themselves
		if ( $debug ) error_log(__FUNCTION__ . " DEBUG: same link break"); 
		break;
	}

	$link = $redirect;
	if ( $debug ) error_log(__FUNCTION__ . " DEBUG: link now {$link}"); 
  }

  if ( false === strpos( $link, 'http' ) )
	$link = null;

  set_transient( $key, $link, 60 );

  if ( $debug ) error_log(__FUNCTION__ . " DEBUG: returning {$link}"); 

  return $link;
}
endif;


/**
 * Show admin notice if this link is already entered on another post.
 */
if ( ! function_exists( 'maddos_check_for_duplicate_link' ) ) :
function maddos_check_for_duplicate_link( $post_id, $link ) {

	$args = array(
		'post__not_in'	=> array( $post_id ),
		'meta_query'	=> array(
			'relation'	=> 'OR',
			array(
				'key'	=> 'site_url',
				'value'	=> $link
			),
			array(
				'key'	=> '_maddos_true_url',
				'value'	=> $link
			),
		),
	);
	$posts = new WP_Query( $args );
	while ( $posts->have_posts() ) {
		$posts->the_post();

		$id = get_the_ID();

		// sometimes WordPress returns bogus post result. Doublecheck fields!
  		$site_url = function_exists( 'get_field' ) ? get_field( 'site_url', $id ) : null;
        	$site_link = maddos_site_url_check( $site_url, false );
		if ( $link != $site_url && $link != $site_link ) continue;

		$msg = sprintf( __( "Maddos Warning! Post <a target='blank' href='%s'>%s</a> just published - found site url %s also exists in post <a target='_blank' href='%s'>%s</a>", 'maddos' ),
			get_post_permalink( $post_id ),
			$post_id,
			esc_url( $site_link ),
			get_post_permalink( $id ),
			$id
		);
		maddos_add_admin_notice( "<div class='notice notice-warning'><p>" . $msg . "</p></div>" );
	}
}
endif;


/**
 * Trigger any Auto Detect Maddos features when post is saved
 */
add_action('save_post', 'maddos_auto_detect', 99, 3 );		// used when saving
if ( ! function_exists( 'maddos_auto_detect' ) ) :
function maddos_auto_detect( $post_id, $post, $update ) {

	if ( ! function_exists( 'get_field' ) ) return;

	if ( wp_is_post_revision( $post_id ) ) return;

  	$debug = apply_filters( 'maddos_debug', false );

	$status = $post->post_status;
	if ( $status === 'trash' || $status === 'auto-draft' ) return;

	if ( false === get_transient( substr( 'maddos-' . str_replace( '-beta', '', ot_get_option( 'user_id' ), $beta ), 0, 45 ) ) ) {
		$msg = sprintf( __( "Saving post: %s denied - License not Activated.", 'maddos' ), __FUNCTION__ );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return;
	}

	maddos_extend_execution_time();

	// do we have a valid site_url?
  	$site_url = function_exists( 'get_field' ) ? get_field( 'site_url', $post_id ) : null;
	// handle any cloaking/redirects to get true url
        $link = maddos_site_url_check( $site_url, true );
        if ( !$link ) return;

	// if cloak/redirect, save the true url for easier searching for dupes:
	if ( $link != $site_url ) {
		add_post_meta( $post_id, '_maddos_true_url', $link, true ) || update_post_meta( $post_id, '_maddos_true_url', $link );
	}

	// check for duplicate site_url, even cloaked, on post publish
	if ( $post->post_modified_gmt === $post->post_date_gmt ) {
		maddos_check_for_duplicate_link( $post_id, $site_url, $link );
	}

	// get auto_detect fields
        $auto_detect = function_exists( 'get_field' ) ? get_field( 'auto_detect', $post_id ) : null;

	// send initial request for screenshot. may take some time. do other stuff and check results below.
	if ( ! empty( $auto_detect ) && in_array( 'screenshot', $auto_detect ) && ! has_post_thumbnail( $post_id ) ) {
		$mshot = maddos_screenshot( $post_id, $link );
		$time = -time();
	}
	else {
		$mshot = null;
	}

	// while mshots generates a screenshot, we can do other stuff while waiting.

	// check for site_icon detect
	if ( ! empty( $auto_detect ) && in_array( 'site_icon', $auto_detect ) && ! get_field( 'site_icon', $post_id ) ) {
		maddos_site_icon( $post_id, $link );
	}

	// check for mobile_friendly detect
	if ( ! empty( $auto_detect ) && in_array( 'mobile_friendly', $auto_detect ) && ! get_field( 'mobile_friendly', $post_id ) ) {
		maddos_mobile_friendly( $post_id, $link );
	}

	// check for title, desc and tags?
	$auto_title = $auto_desc = $auto_tags = false;
	$post_change = array();
	if ( ! empty( $auto_detect ) && in_array( 'title', $auto_detect ) && empty( $post->post_title ) ) {
		$auto_title = true;
	}
	if ( ! empty( $auto_detect ) && in_array( 'description', $auto_detect ) && empty( $post->post_content ) ) {
		$auto_desc = true;
	}
	if ( ! empty( $auto_detect ) && in_array( 'tags', $auto_detect ) && ! has_tag( $post ) ) {
		$auto_tags = true;
	}

	if ( $auto_title ) {
		$page = file_get_contents( $link, false, null, 0, 1000 ); // 1st 1000 bytes should have title
		$title = preg_match('/<title[^>]*>(.*?)<\/title>/ims', $page, $match) ? $match[1] : null;
		if ( ! empty( $title ) ) {
			$post_change['post_title'] = $title;
		}
  		if ( $debug ) error_log(__FUNCTION__ . " DEBUG: ({$link}) found title: " . $title);
	}
	if ( $auto_desc || $auto_tags ) {
		$meta_tags = $auto_desc || $auto_tags ? get_meta_tags( $link ) : array();
  		if ( $debug ) error_log(__FUNCTION__ . " DEBUG: ({$link}) meta_tags: " . print_r($meta_tags,true));
		if ( $auto_desc && isset( $meta_tags['description'] ) ) {
  			if ( $debug ) error_log(__FUNCTION__ . " DEBUG: ({$link}) found desc: " . $meta_tags['description']);
			$post_change['post_content'] = $meta_tags['description'];
		}
		if ( $auto_tags && isset( $meta_tags['keywords'] ) ) {
			$tags = $meta_tags['keywords'];
			$tags = str_replace( array("\n", "\r\n"), ",", $tags );
  			if ( $debug ) error_log(__FUNCTION__ . " DEBUG: ({$link}) found tags/keywords: " . $tags );
		}
	}
	// update post if needed
	if ( ! empty( $post_change ) ) {
		$post_change['ID'] = $post_id;
		remove_action( 'save_post', 'maddos_auto_detect', 99 );
		$response = wp_update_post( $post_change );
		add_action( 'save_post', 'maddos_auto_detect', 99 );
  		if ( is_wp_error( $response ) ) {
			$msg = sprintf( __( "%s WP Error on wp_update_post with args %s: %s", 'maddos' ), __FUNCTION__, print_r($post_change,true), $response->get_error_message() );
			maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
			return;
		}
	}
	// append tags if needed
	if ( isset( $tags ) && ! empty( $tags ) ) {
		$response = wp_set_post_tags( $post_id, $tags, true );
  		if ( is_wp_error( $response ) ) {
			$msg = sprintf( __( "%s WP Error on wp_set_post_tags with args %s: %s", 'maddos' ), __FUNCTION__, print_r($tags,true), $response->get_error_message() );
			maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
			return;
		}
	}

	// now with the screenshot request url made from above, check for results
	if ( $mshot ) {
		// we're giving the above request ~10 seconds to get the screenshot...
		$time += time();
		$time = $time < 10 ? 10 - $time : 0;
		// in case user wants to add even more time
		$time = apply_filters( 'maddos_mshots_wait', $time);
		if ( $time ) {
			sleep( $time );
		}
		maddos_screenshot( $post_id, $link, $mshot );
	}
}
endif;



/**
 * Fetch a site icon automatically from Google favicons API
 */
if ( ! function_exists( 'maddos_site_icon' ) ) :
function maddos_site_icon( $post_id, $link = null, $default_check = false ) {

  $debug = apply_filters( 'maddos_debug', false );

	$common_icons = apply_filters( 'maddos_common_icons', array( 
		'3ca64f83fdcf25135d87e08af65e68c9' => '/images/internet.png',
		'af8f520386d91238ce169993d635aca7' => '/images/default.png',
		'28b959f8750f0e19f688de8bba118aa5' => '/images/blank.png'
	) );

	// only check for new icon if current icon is 'common'
	if ( true === $default_check ) {
  		$current_icon = function_exists( 'get_field' ) ? get_field( 'site_icon', $post_id ) : null;
		if ( ! empty( $current_icon ) ) {
			$current_bits = file_get_contents( $current_icon );
			$current_md5 = md5( $current_bits );
			if ( ! isset( $common_icons[$current_md5] ) ) {
				return;
			}
		}
	}

	if ( empty( $link ) ) {
  		$site_url = function_exists( 'get_field' ) ? get_field( 'site_url', $post_id ) : null;
		// handle any cloaking/redirects to get true url
        	$link = maddos_site_url_check( $site_url );
	}
	if ( empty( $link ) ) {
		$msg = sprintf( __( "%s link is empty, please contact support.", 'maddos' ), __FUNCTION__ );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return;
	}
	$parse = parse_url( $link );
  	if ( $debug ) error_log(__FUNCTION__ . " DEBUG: parsing ({$link}) into: " . print_r($parse,true) ); 

	if ( empty( $parse ) || ! isset( $parse['host'] ) ) {
		$msg = sprintf( __( "%s: Unable to parse_url( %s ) - please contact support.", 'maddos' ), __FUNCTION__, $link );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return;
	}
	$name = substr( $parse['host'], 0, -4 ) . ".png";
	$name = apply_filters( 'maddos_site_icon_name', $name, $link );

	$icon = apply_filters( 'maddos_site_icon_url', null, $post_id, $link );
	$icon = empty( $icon ) ? "http://www.google.com/s2/favicons?domain=" . $parse['scheme'] . "://" . $parse['host'] : $icon;

  	if ( $debug ) error_log(__FUNCTION__ . " DEBUG: icon check is ({$icon})"); 

  	// some websites do not recognize the default WordPress user-agent, so use a dummy agent...
  	$args = array( 'user-agent' => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.8; rv:20.0) Gecko/20100101 Firefox/20.0' );
	$response =  wp_remote_get( esc_url( $icon ), $args );
  	if ( is_wp_error( $response ) ) {
		$msg = sprintf( __( "%s WP Error on %s: %s", 'maddos' ), __FUNCTION__, esc_url( $icon ), $response->get_error_message() );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return;
	}
  	else if ( empty( $response['body'] ) ) {
		$msg = sprintf( __( "%s Empty response for %s", 'maddos' ), __FUNCTION__, esc_url( $icon ) );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return;
	}

  	if ( $debug ) error_log(__FUNCTION__ . " DEBUG: icon response: " . print_r($response,true) );

	$img_bits = wp_remote_retrieve_body( $response );

	$md5 = md5( $img_bits );

	// check for common icons. just use a single, local copy.
	if ( isset( $common_icons[$md5] ) ) {
		$icon = get_theme_file_uri( $common_icons[$md5] );
		if ( is_ssl() ) $icon = set_url_scheme( $icon );
		// this is the ACF key for site_icon:
		update_field( 'field_5612c09af2cef', $icon, $post_id );
		return;
	}

	$upload = wp_upload_bits( $name, null, $img_bits );

	if ( $upload['error'] ) {
		$msg = sprintf( __( "%s wp_upload_bits error for %s: %s", 'maddos' ), __FUNCTION__, $name, $upload['error'] );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return;
	}
	
	// in case user wants to do something
	$upload = apply_filters( 'maddos_site_icon_upload', $upload );


	if ( isset( $upload['url'] ) ) {
		if ( is_ssl() ) $upload['url'] = set_url_scheme( $upload['url'] );
		// this is the ACF key for site_icon:
		update_field( 'field_5612c09af2cef', $upload['url'], $post_id );
		// add to media library
		$filetype = wp_check_filetype( basename( $upload['file'] ), null );
		$attachment = array(
			'guid'           => $upload['file'],
			'post_mime_type' => $filetype['type'],
			'post_title'     => basename( $upload['file'] ),
			'post_content'   => '',
			'post_status'    => 'inherit'
		);

		$attach_id = wp_insert_attachment( $attachment, $upload['file'], $post_id );
		if ( $attach_id ) {
			// Make sure that this file is included, as wp_generate_attachment_metadata() depends on it.
			require_once( ABSPATH . 'wp-admin/includes/image.php' );
			// generate the metadata for the attachment
			wp_update_attachment_metadata( $attach_id, wp_generate_attachment_metadata( $attach_id, $upload['file'] ) );
		}
		else {
			$msg = sprintf( __( "%s could not attach file %s to media library.", 'maddos' ), __FUNCTION__, $upload['file'] );
			maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		}
	}
}
endif;

function maddos_decode( $value ) {
	$func = 'base64' . '_decode';
	return $func( $value );
}

/**
 * Determine if site if mobile friendly using Google mobileFriendlyTest
 * https://developers.google.com/webmaster-tools/search-console-api/reference/rest/v1/urlTestingTools.mobileFriendlyTest/run
 */
if ( ! function_exists( 'maddos_mobile_friendly' ) ) :
function maddos_mobile_friendly( $post_id, $link ) {

	$debug = apply_filters( 'maddos_debug', false );

	// call google API to see if mobile ready
	$apiKey = apply_filters( 'maddos_google_api_key', 'AIzaSyDowN39z9BRxzXVlt_3Iw7ejKbn9nAxcz8' );
	$url = "https://searchconsole.googleapis.com/v1/urlTestingTools/mobileFriendlyTest:run?key={$apiKey}" ;
  	$args = array( 
		'timeout'		=> 30,
		'body' => array( 
			'requestScreenshot'	=> false,	// might be used in future if mshots goes away
			'url'			=> $link
		)
	);
  	if ( $debug ) error_log(__FUNCTION__ . " DEBUG: ({$url}) with args: " . print_r($args,true)); 

	$response = wp_remote_post( esc_url( $url ), $args );

	if ( is_wp_error( $response ) ) {
		$msg = sprintf( __( "%s WP Error on %s: %s", 'maddos' ), __FUNCTION__, esc_url( $url ), $response->get_error_message() );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
  		if ( $debug ) error_log(__FUNCTION__ . " DEBUG result: " . $response->get_error_message() );
		return;
	}

	$result = json_decode( $response['body'], true );
  	if ( $debug ) error_log(__FUNCTION__ . " DEBUG result: " . print_r($result,true));

	$mf = false;
       	if ( empty( $result['mobileFriendliness'] ) ) {
			$msg = sprintf( __( "%s Did not receive expected response from %s : result [%s]", 'maddos' ), __FUNCTION__, esc_url( $url ), print_r($result,true) );
			maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
	}
	else if ( $result['mobileFriendliness'] === 'MOBILE_FRIENDLY' ) 
		$mf = true;

	// this is the ACF key for mobile_friendly:
        update_field( 'field_5612c0a7f2cf0', (bool) $mf, $post_id );
}
endif;

/**
 * Capture a screenshot of the webpage using WordPress.com mShots API and set as featured image
 * if an img url is passed in, that will be used instead of generating a screenshot
 */
if ( ! function_exists( 'maddos_screenshot' ) ) :
function maddos_screenshot( $post_id, $link, $img = null ) {

	$debug = apply_filters( 'maddos_debug', false );

	if ( ! function_exists( 'download_url' ) ) {
		include_once( ABSPATH . 'wp-admin/includes/image.php' );
		include_once( ABSPATH . 'wp-admin/includes/file.php' );
		include_once( ABSPATH . 'wp-admin/includes/media.php' );
	}

	// initial request
	if ( ! $img ) {
		$width = apply_filters( 'maddos_screenshot_width', 1280 );
        	$link = urlencode( $link );
		$mshot = 'http://s.wordpress.com/mshots/v1/' . $link . '?w=' . $width . ".jpg";
  		if ( $debug ) error_log(__FUNCTION__ . " DEBUG: ({$mshot})"); 
		$tmp = download_url( $mshot );
		if ( is_wp_error( $tmp ) ) {
			$msg = sprintf( __( "%s WP Error on %s: %s", 'maddos' ), __FUNCTION__, esc_url( $mshot ), $tmp->get_error_message() );
			maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
			return 0;
		}
		unlink( $tmp );
		return $mshot;
	}
	else $mshot = $img;

	// second call, check initial request results
	// THREE different ways of downloading the screenshot... 
	$attach_id = maddos_screenshot_download_memory( $post_id, $mshot );
	if ( ! $attach_id ) {
		$msg = __( "Attempting screenshot download method #2", 'maddos' );
		maddos_add_admin_notice( "<div class='notice notice-warning'><p>" . $msg . "</p></div>" );
		$attach_id = maddos_screenshot_download_tmp( $post_id, $mshot );
	}
	if ( ! $attach_id ) {
		$msg = __( "Attempting screenshot download method #3", 'maddos' );
		maddos_add_admin_notice( "<div class='notice notice-warning'><p>" . $msg . "</p></div>" );
		$attach_id = maddos_screenshot_download_sideload( $post_id, $mshot );
	}

	if ( $attach_id ) {
		// set as featured image
		if ( ! set_post_thumbnail( $post_id, $attach_id ) ) {
			$msg = sprintf( __( "%s set_post_thumbnail( postID %s, attachID %s ) returned failure. Unknown reason.", 'maddos' ), __FUNCTION__, $post_id, $attach_id );
			maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
			return;
		}
	}
}

function maddos_screenshot_download_tmp( $post_id, $mshot ) {
	$tmp = download_url( $mshot );
	if ( is_wp_error( $tmp ) ) {
		$msg = sprintf( __( "%s WP Error on %s: %s", 'maddos' ), __FUNCTION__, esc_url( $mshot ), $tmp->get_error_message() );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return 0;
	}

	// okay we now have the $tmp location of the screenshot, so copy it to the WP upload directory

	$post_title = get_the_title( $post_id );
	$basefile = preg_replace("/[^a-zA-Z0-9]+/", "", $post_title ) . '.jpg';
	$basefile = apply_filters( 'maddos_screenshot_name', $basefile, $post_id );
	$wp_upload_dir = wp_upload_dir();
	$filename = $wp_upload_dir['path'] . '/' .  $basefile;

	if ( ! rename ( $tmp, $filename ) ) {
		$msg = sprintf( __( "%s could not move %s to %s", 'maddos' ), __FUNCTION__, esc_url( $tmp ), esc_url( $filename ) );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		unlink( $tmp );
		return 0;
	}

	$filetype = wp_check_filetype( basename( $filename ), null );
	if ( empty( $filetype ) ) {
		$msg = sprintf( __( "%s could not determine filetype of %s", 'maddos' ), __FUNCTION__, esc_url( $filename ) );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return 0;
	}

	// now create and insert attachment into WP

	$attachment = array(
		'guid'           => $wp_upload_dir['url'] . '/' . basename( $filename ),
		'post_mime_type' => $filetype['type'],
		'post_title'     => preg_replace( '/\.[^.]+$/', '', basename( $filename ) ),
		'post_content'   => '',
		'post_status'    => 'inherit'
	);
	$attachment = apply_filters( 'maddos_screenshot_attachment', $attachment, $filename );
	if ( ! file_exists( $filename ) ) {
		$msg = sprintf( __( "%s file %s was removed by unknown code.", 'maddos' ), __FUNCTION__, esc_url( $filename ) );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return 0;
	}
	$attach_id = wp_insert_attachment( $attachment, $filename, $post_id );

	// now generate metadata for attachment
	if ( $attach_id ) {
		// Make sure that this file is included, as wp_generate_attachment_metadata() depends on it.
		require_once( ABSPATH . 'wp-admin/includes/image.php' );
		// generate the metadata for the attachment
		$metadata = apply_filters( 'maddos_screenshot_metadata', wp_generate_attachment_metadata( $attach_id, $filename ), $attach_id );
		if ( empty( $metadata ) ) {
			$msg = sprintf( __( "%s wp_generate_attachment_metadata( attach_id %s, filename %s ) returned failure. Unknown reason.", 'maddos' ), __FUNCTION__, $attach_id, esc_url( $filename ) );
			maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
			return 0;
		}
		wp_update_attachment_metadata( $attach_id, $metadata );
		return $attach_id;

	}
	else {
		$msg = sprintf( __( "%s could not attach file %s to media library.", 'maddos' ), __FUNCTION__, esc_url( $filename ) );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return 0;
	}
}

//===========================================================

function maddos_screenshot_download_memory( $post_id, $mshot ) {
  	// some websites do not recognize the default WordPress user-agent, so use a dummy agent...
  	$args = array( 'user-agent' => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.8; rv:20.0) Gecko/20100101 Firefox/20.0' );
	$response = wp_remote_get( esc_url( $mshot ), $args );
	if ( is_wp_error( $response ) ) {
		$msg = sprintf( __( "%s WP Error on %s: %s", 'maddos' ), __FUNCTION__, esc_url( $mshot ), $response->get_error_message() );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return 0;
	}
	$post_title = get_the_title( $post_id );
	$basefile = preg_replace("/[^a-zA-Z0-9]+/", "", $post_title ) . '.jpg';
	$basefile = apply_filters( 'maddos_screenshot_name', $basefile, $post_id );
	$upload = wp_upload_bits( $basefile, '', $response['body'] );
	if ( $upload['error'] ) {
		$msg = sprintf( __( "%s WP Error in wp_upload_bits: %s", 'maddos' ), __FUNCTION__, $upload['error'] );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return 0;
	}
	$filetype = wp_check_filetype( basename( $upload['file'] ), null );

	$attachment = array(
		'post_title'		=> $basefile,
		'post_mime_type' 	=> $filetype['type'],
		'post_content'		=> '',
		'post_status'		=> 'inherit'
	);
	$attach_id = wp_insert_attachment( $attachment, $upload['file'], $post_id );

	if ( $attach_id ) {
		// Make sure that this file is included, as wp_generate_attachment_metadata() depends on it.
		require_once( ABSPATH . 'wp-admin/includes/image.php' );
		// generate the metadata for the attachment
		$metadata = apply_filters( 'maddos_screenshot_metadata', wp_generate_attachment_metadata( $attach_id, $upload['file']), $attach_id );
		if ( empty( $metadata ) ) {
			$msg = sprintf( __( "%s wp_generate_attachment_metadata( attach_id %s, filename %s ) returned failure. Unknown reason.", 'maddos' ), __FUNCTION__, $attach_id, esc_url( $upload['file'] ) );
			maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
			return 0;
		}
		wp_update_attachment_metadata( $attach_id, $metadata );
		return $attach_id;
	}
	else {
		$msg = sprintf( __( "%s could not attach file %s to media library.", 'maddos' ), __FUNCTION__, esc_url( $upload['file'] ) );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return 0;
	}
}

//===========================================================

function maddos_screenshot_download_sideload( $post_id, $mshot ) {
	$post_title = get_the_title( $post_id );
	$basefile = preg_replace("/[^a-zA-Z0-9]+/", "", $post_title ) . '.jpg';
	$basefile = apply_filters( 'maddos_screenshot_name', $basefile, $post_id );

	$response = media_sideload_image( $mshot, $post_id, $basefile );

	if ( is_wp_error( $response ) ) {
		$msg = sprintf( __( "%s WP Error: %s", 'maddos' ), __FUNCTION__, $response->get_error_message() );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
		return 0;
	}

	$attachments = get_posts(array('numberposts' => '1', 'post_parent' => $post_id, 'post_type' => 'attachment', 'post_mime_type' => 'image', 'order' => 'DESC'));
	if( sizeof( $attachments ) > 0 ) {
		return $attachments[0]->ID;
	}
}

endif;



/*
 * Hook into Advanced Custom Fields plugin to prepend site icon
 */
add_filter('acf/load_field/name=site_url', 'maddos_acf_load_field', 10, 1);
add_filter('acf/load_field/name=site_icon', 'maddos_acf_load_field', 10, 1);
add_filter('acf/load_field/name=additional_icon', 'maddos_acf_load_field', 10, 1);
if ( ! function_exists( 'maddos_acf_load_field' ) ) :
function maddos_acf_load_field( $field ) {
	$post_id = get_the_ID();
	$value = get_post_meta( $post_id, $field['name'], true );

	if ( ! $value ) return $field;

	// site_url value - link to the site
	if ( $field['name'] === 'site_url' ) {
		$icon = get_theme_file_uri( "/images/internet.png" );
		$field['append'] = '<a target="_blank" href="' . $value . '"><img width="16" height="16" src="' . $icon . '" data-pin-nopin="true"></a>';
	}
	// site_icon or additional icon value with a valid url
	else if ( filter_var( $value, FILTER_VALIDATE_URL ) ) {
		$field['prepend'] = '<img width="16" height="16" src="' . $value . '" data-pin-nopin="true">';
	}
	// site_icon or additional icon value without a valid url
	else {
		$field['prepend'] = $value;
	}
	return $field;
}
endif;


/*
 * Warn if server config not enough for post_max_size or max_input_vars
 * MARK: this does not get current post size or input vars for some reason...
 */
add_action( 'admin_notices', 'maddos_check_server_config' );
function maddos_check_server_config() {
	$debug = apply_filters( 'maddos_debug', false );

	$post_max_size = ini_get( 'post_max_size' );
	$current_post_size = isset( $_SERVER['CONTENT_LENGTH'] ) ? $_SERVER['CONTENT_LENGTH'] : 0;
  	if ( $debug && $current_post_size ) {
		error_log( sprintf( 'Post max size is %s, current post size is %s', $post_max_size, $current_post_size ) );
	}
	if ( $current_post_size >= $post_max_size ) {
		$msg = sprintf( esc_html__( 'PHP cannot handle this amount of data, suggest raising post_max_size in php.ini. Post max size is %s, current post size is %s', 'maddos' ), $post_max_size, $current_post_size );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
	}

	$max_input_vars = ini_get( 'max_input_vars' );
	$current_input_vars = isset( $_POST ) ? count($_POST, COUNT_RECURSIVE) : 0;
	$current_input_vars = count($_POST) + count($_GET) + count($_COOKIE);
	if ( $debug && $current_input_vars ) {
		error_log( sprintf( 'Maddos debug - Max input vars is %s, current input vars is %s', $max_input_vars, $current_input_vars ) );
	}
	if ( $current_input_vars >= $max_input_vars ) {
		$msg = sprintf( esc_html__( 'PHP cannot handle this amount of variables, suggest raising max_input_vars in php.ini. Max input vars is %s, current input vars is %s', 'maddos' ), $max_input_vars, $current_input_vars );
		maddos_add_admin_notice( "<div class='error'><p>" . $msg . "</p></div>" );
	}


}

/*
 * Warn if no license information
 */
add_action( 'ot_after_theme_options_save', 'maddos_check_license' );
function maddos_check_license( $options ) {
	if ( false === ot_get_option( 'user_id' ) || false === ot_get_option( 'user_code' ) ) {
		$msg = __( "Maddos Warning! Please enter License information to unlock this theme and get automatic updates.", 'maddos' );
		maddos_add_admin_notice( "<div class='notice notice-warning'><p>" . $msg . "</p></div>" );
	}

maddos_check_server_config();
	// after save, check license and for updates
	maddos_tuc_setup();
}

function maddos_license_status() {
	$name = str_replace( '-beta', '', ot_get_option( 'user_id' ), $beta );
	$key = 'maddos-' . $name;
	$key = substr( $key, 0, 45 );
	$transient = get_transient( $key );
	$str = $transient ? "<br>" . $transient : null;
	return $str;
}

/*
 * Extend script time if needed. Only works if running mod_php (default module on Apache).
 */
if ( ! function_exists( 'maddos_extend_execution_time' ) ) :
function maddos_extend_execution_time() {
	set_time_limit( 0 );
}
endif;


/*
 * Intercept custom meta field save and xfer it to ACF field automaticlly.
 * Specifically so we can use a User Submitted Posts plugin to import a post (plugin only supports custom field not ACF)
 */
add_filter('update_post_metadata', 'maddos_metadata_check', 9999, 5);
function maddos_metadata_check( $check, $object_id, $meta_key, $meta_value, $prev_value ) {

	if ( $meta_key === 'website_url' ) {
		remove_filter( current_filter(), __FUNCTION__ ); // avoid infinite loop when updating meta

		// convert custom meta field to actual ACF field
		update_field( 'field_560c0e2cc0edc', $meta_value, $object_id );
		// don't save custom meta - we saved as ACF

		// also set auto detect defaults to true
		$auto = apply_filters( 'maddos_user_post_auto_detect', array( 'site_icon', 'mobile_friendly', 'screenshot', 'title', 'description', 'tags' ) );
		update_field( 'field_570d604a53c64', $auto, $object_id );

		// do not continue the update_post_metadata
		return false;

		// note this does not trigger auto-detect since this code is run after save_post already triggered.
	}

	// return null to continue the update_post_metadata
	return null;
}


function maddos_get_target( $new_tab = true ) {

	$target = true === $new_tab ? 'target="_blank"' : '';

	$metabox_options = function_exists( 'get_field' ) ? get_field( 'options' ) : array();
	$target = $metabox_options && in_array( 'new_tab', $metabox_options ) ? 'target="_blank"' : $target;

	return $target;
}

function maddos_get_rel( $nofollow = true, $target = '' ) {

	$rel = true === $nofollow ? 'nofollow' : '';

	$metabox_options = function_exists( 'get_field' ) ? get_field( 'options' ) : array();
	$rel = $metabox_options && in_array( 'no_follow', $metabox_options ) ? 'nofollow' : $rel;

	$rel .= true === ! empty( $target ) ? ' noopener' : '';
	$rel = ! empty( $rel ) ? 'rel="' . trim( $rel ) . '"' : '';

	return $rel;
}

function maddos_get_post_categories( $postID = 0 ) {
	if ( ! $postID ) {
		$postID = get_the_ID();
	}
	if ( ! $postID ) {
		return '';
	}
	$categories = get_the_category();
	$categories = apply_filters( 'maddos_post_categories', $categories, $postID );
	$catNames = array();
	foreach ( $categories as $category ) {
	        if ( isset( $category->name ) ) {
			$catNames[] = sprintf( "<a href='%s'>%s</a>", 
				get_category_link( $category->term_id ),
				$category->name
			);
		}
	}
       	$cat_name = empty( $catNames ) ? '' : implode( ', ', $catNames );
	$cat_name = apply_filters( 'maddos_post_categories_output', $cat_name, $postID );
	return $cat_name;
}

?>
